
/****************************************************************************/
/* File: tuner.c                                                            */
/* Author: Gary M. Segal                                                    */
/* Copyright (C) 2005, Freescale Semiconductor INC                          */
/****************************************************************************/
/* Description:
/*   This file implements the control functions for the Freescale Silicon
/* tuners, MC44S802A & MC44S803
/*
/* User Functions:
/* tuner_init (): This function initializes the tuner hardware and control
/*     software.  It must be called as part of system initialization,
/*     before any other tuner functions are called.
/* tune (frequency, frequency_type): This function tunes the chip to the
/*     desired frequency.  It accepts either a center frequency or a picture
/*     carrier.  The picture carrier offset is defined in tuner.h.
/*
/* Dependancies:
/* These routines require an underlying I2C communication channel.  Calls
/* are made to the following functions:
/* i2c_xmit (address, data, data_len): called by various tuner function to
/*     communicate with the tuner.
/* i2c_rcv (adress, data, data_len): called by various tuner function to
/*     communicate with the tuner.
/*
/* The global structure g_regs is used to hold information about each of
/* the registers in the tuner including register address, register length
/* and register data.  It is used for both reading and writing to the
/* tuner.
/*
/* Note that the file tuner_settings.h contains system parameters that
/* must be set based on the hardware design.  This header file must
/* be set to indicate the reference clock frequency, the desired
/* first IF, and the output IF.
/*
/****************************************************************************/
/* revision history:                                                        */
/*                                                                          */
/* Date      Version  Comments                                      Initial */
/* ------------------------------------------------------------------------ */
/* 1/18/05   1.0      Intial General Release                        gms     */
/* 3/18/05   1.1      Corrected n & r divider calculations.  Added  gms     */
/*                    i2c_xmit_tuner_reg, i2c_read_tuner_reg and            */
/*                    modified data structures to support these routines.   */
/*                    Added power calculation routine.                      */
/* 3/29/05   1.2      Udated initilization code to setup ref osc    gms     */
/*                    register                                           .  */
/* 4/8/05    1.3      Added delay for ref osc to settle.  Setup     gms     */
/*                    all other registers.  Added power-up prior to         *
/*                    setting digital tune prescaller register (XO).        */
/* 4/13/05   1.4      Renamed functions to include part number.     gms     */
/*                    Added auto-initilziation.                             */
/* 4/14/05   1.5      Changed from LNA gain table to RF AGC table   gms     */
/*                    with interpolation.  Changed input to tune function   */
/*                    to use a float holding frequency in MHz.              */
/* 7/12/05   1.6      Updated for MC44S802A                         gms     */
/* 10/25/05  1.7      Updated to support 802A and 803.  Added       gms     */
/*                    params to tuner_init.                                 */
/* 12/13/05  1.8      Updated tuner init values (LNABias) and       gms     */
/*                    removed unused code.                                  */
/* 1/10/06   1.9      Updated circuit adjust values.                gms     */
/* 2/8/06    1.10     Updated ref osc power up wait time to 20mSec  gms     */
/* 6/1/06    1.11     New version started for 803.  RF/IF LNA AGC   gms     */
/*                    loop added.                                           */
/* 8/7/06    1.12     Added averaging for RF/IF LNA AGC loops.      gms     */
/*                    Moved REFOSC register init just after tuner           */
/*                    power-on in tuner_init.                               */
/*                                                                          */
/****************************************************************************/

#include "FS_mcf5307.h"
#include "FS_tuner.h"
#include "FS_tuner_settings.h"
#include "FS_tuner_globals.h"
#include "FS_i2c.h"
#include "FS803A.h"
#include "af901x.h"

extern TUNER_REGS_T tuner_regs;

// digital agc
extern int		dagc_rf_max;
extern int		dagc_rf_min;
extern int		dagc_if_max;
extern int		dagc_if_min;

/****************************************************************************
* Function Name: i2c_xmit_tuner_reg
*
* Description:
* This routine transmits a tuner register to the tuner using the I2C bus.
*
* Parameters:
*	tuner_register: A tuner register structure.  The structure
*        contains the information needed to address the register as
*        well as the data to transmit.
*
* Output Conditions:
*    If no error occurs, the register data has been sent to the tuner over
*    the I2C bus.
*
* Return type: integer
* Return values:
*      >0 : an I2C error occured.  The returned value is an I2C error code.
*      0 : the data was sucesfully transmited.
*
*****************************************************************************/
int MC44S803_i2c_xmit_tuner_reg (GENERIC_STRUCT_T *tuner_reg)
{
	int ret;
	
	ret = FALSE;
	
	/* Frist, set the address field of the register to the correct address */
	tuner_reg->data.bits.Addr = tuner_reg->address;
			
	/* Now, transmit the register */
#if BIG_ENDIAN
    ret = MC44S803_i2c_xmit (TUNER_I2C_ADDRESS, tuner_reg->size,
                        &(tuner_reg->data.bytes[tuner_reg->dataoffset]));
#else
    ret = MC44S803_i2c_xmit (TUNER_I2C_ADDRESS, tuner_reg->size,
                        &(tuner_reg->data.bytes[tuner_reg->size - 1]));
#endif

	return (ret);
}


/****************************************************************************
* Function Name: i2c_read_tuner_reg
*
* Description:
* This routine reads a tuner register to the tuner using the I2C bus.
*
* Parameters:
*	tuner_register: A tuner register structure.  The structure
*        contains the information needed to address the register as
*        well as the location to hold the received data.
*
* Output Conditions:
*    If no error occurs, the register data has been read to the tuner over
*    the I2C bus and loaded into the tuner_register data structure.
*
* Return type: integer
* Return values:
*      >0 : an I2C error occured.  The returned value is an I2C error code.
*      0 : the data was sucesfully transmited.
*
*****************************************************************************/
int MC44S803_i2c_read_tuner_reg (GENERIC_STRUCT_T *tuner_reg)
{
	int ret;
	int register_address;
	
	ret = FALSE;

	/* Determine the address of the requested register and load it into the
	   local copy of the data address register */
	register_address = tuner_reg->address;
	g_regs.data_addr.data.bits.data_addr = register_address;
	
	/* Send the data address register to the tuner. */
	if (ret=MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.data_addr))) {	    
	} else {
		/* Read back the requested register */
#if BIG_ENDIAN
		ret = MC44S803_i2c_rcv (TUNER_I2C_ADDRESS, tuner_reg->size,
						&(tuner_reg->data.bytes[tuner_reg->dataoffset]));
#else
		ret = MC44S803_i2c_rcv (TUNER_I2C_ADDRESS, tuner_reg->size,
						&(tuner_reg->data.bytes[tuner_reg->size - 1]));
#endif
	}
	return (ret);
}



/****************************************************************************
* Function Name: locate_optimal_settings
*
* Description:
* This routine locates the approrpiate entry in the optimal settings table
* for a given frequency.  It returns the index into the table for the entry.
* If the input frequency is out of range, -1 is returned.
*
* Note that the optimal settings table is arranged so that a record applies
* to frequencies greater than or equal to the record's frequency, and less
* than the next record's frequency.  This routine scans the table by starting
* at the low end and checking to see if the desired frequency is less than
* the current record.  Thus, when this condition is true, the desired record
* is the one prior to the current record.
*
* Parameters:
*	freq: Double percision floating point number represeting the center
*      frequency of the desired channel in Hz
* Return type: integer
* Return values:
*      >= 0: values greater than or equal to zero indicate the offset into
*             g_optimal_settings table that corresponds to the desired
*             frequency.
*      -1: indicates that the desired frequency is out of range
*
*****************************************************************************/
int MC44S803_locate_settings (int freq)
{
    int i;	/* index counter for g_optimal_settings_table */
    int done;	/* search flag */


    i = 0;
    done = FALSE;

    /* First, check if the frequency is below the minimum entry */
    if (freq < g_tuner_settings_table[0].freq) {
    	i = FREQ_TO_LOW;
    	done = TRUE;
    }

    while (!done) {
    	/* First, make sure we're not at the end of the list */
    	if (g_tuner_settings_table[i].freq == END_FREQ) {
    		done = TRUE;
    		i--; /* Decrement the counter to use the previous entry */
    	} else {
    	/* We're not at the end of the list, so check the current entry */
    	    if (freq <= g_tuner_settings_table[i].freq) {
    	    	/* Found a match, so stop searching */
    	    	done = TRUE;
    	    } else {
    	    	/* No match found, go to the next recrod */
	    	    i++;
       	    }
    	}
    }
    return (i);
}




/****************************************************************************
* Function Name: set_powerdown
*
* Description:
*   This routine programs the power down register in the MC44S802 tuner.
* It loads all bits in the tuner's power down registger with the same
* value, so the chip is either all on or all off.
*
* Parameters:
*	val: the value to be loaded into the power down register's bits.
*
* Return type: integer
* Return values:
*      >0 : an I2C error occured.  The returned value is an I2C error code.
*      0 : the data was sucesfully transmited.
*
*****************************************************************************/
int MC44S803_set_powerdown (int val)
{
	int ret;
	
	g_regs.powerdown.data.bits.Addr = TUNER_POWERDOWN;
	g_regs.powerdown.data.bits.IFPwrDet = val;
	g_regs.powerdown.data.bits.RFPwrDet = val;
	g_regs.powerdown.data.bits.Buf2 = val;
	g_regs.powerdown.data.bits.RefOsc = val;
	g_regs.powerdown.data.bits.AcgAmp = val;
	g_regs.powerdown.data.bits.rsvd1 = POWERDOWN_OFF;
	g_regs.powerdown.data.bits.rsvd2 = POWERDOWN_OFF;
	g_regs.powerdown.data.bits.IF2 = val;
	g_regs.powerdown.data.bits.IF1 = val;
	g_regs.powerdown.data.bits.LNA = val;
	g_regs.powerdown.data.bits.rsvd3 = POWERDOWN_OFF;
	g_regs.powerdown.data.bits.Mix1 = val;
	g_regs.powerdown.data.bits.Synth2 = val;
	g_regs.powerdown.data.bits.CP2 = val;
	g_regs.powerdown.data.bits.Quad = val;
	g_regs.powerdown.data.bits.VCO2 = val;
	g_regs.powerdown.data.bits.Synth1 = val;
	g_regs.powerdown.data.bits.CP1 = val;
	g_regs.powerdown.data.bits.Buf1 = val;
	g_regs.powerdown.data.bits.VCO1 = val;

	ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.powerdown));
	return (ret);

}


/****************************************************************************
* Function Name: tuner_init_globals
*
* Description:
*   This routine tunes initilizes the global tuner register data structure.
* All data fields are cleared.  The data needed to access the registers in
* the MC44S803 are initalized (register address, length, and the offset
* to the data's location within the structure).
*
* Parameters: None.
*
* Return values: None.
*
* Output condition: The global structure g_regs is initalized.
*
*****************************************************************************/
void MC44S803_tuner_init_globals ()
{
    /* First, clear the global register array and set the I2C size/offset data */
    g_regs.powerdown.data.all = 0;
    g_regs.powerdown.size = TUNER_POWERDOWN_LEN;
	g_regs.powerdown.dataoffset = TUNER_REG_BYTES-TUNER_POWERDOWN_LEN;
    
    g_regs.refosc.data.all = 0;
    g_regs.refosc.size = TUNER_REFOSC_LEN;
	g_regs.refosc.dataoffset = TUNER_REG_BYTES-TUNER_REFOSC_LEN;

    g_regs.refdiv.data.all = 0;
    g_regs.refdiv.size = TUNER_REFDIV_LEN;
	g_regs.refdiv.dataoffset = TUNER_REG_BYTES-TUNER_REFDIV_LEN;

    g_regs.mixref.data.all = 0;
    g_regs.mixref.size = TUNER_MIXREF_LEN;
	g_regs.mixref.dataoffset = TUNER_REG_BYTES-TUNER_MIXREF_LEN;

    g_regs.resetso.data.all = 0;
    g_regs.resetso.size = TUNER_RESETSO_LEN;
	g_regs.resetso.dataoffset = TUNER_REG_BYTES-TUNER_RESETSO_LEN;

    g_regs.lo1.data.all = 0;
    g_regs.lo1.size = TUNER_LO1_LEN;
	g_regs.lo1.dataoffset = TUNER_REG_BYTES-TUNER_LO1_LEN;

    g_regs.lo2.data.all = 0;
    g_regs.lo2.size = TUNER_LO2_LEN;
	g_regs.lo2.dataoffset = TUNER_REG_BYTES-TUNER_LO2_LEN;

    g_regs.circuitadj.data.all = 0;
    g_regs.circuitadj.size = TUNER_CIRCUITADJ_LEN;
	g_regs.circuitadj.dataoffset = TUNER_REG_BYTES-TUNER_CIRCUITADJ_LEN;

    g_regs.test.data.all = 0;
    g_regs.test.size = TUNER_TEST_LEN;
	g_regs.test.dataoffset = TUNER_REG_BYTES-TUNER_TEST_LEN;

    g_regs.digtune.data.all = 0;
    g_regs.digtune.size = TUNER_DIGTUNE_LEN;
	g_regs.digtune.dataoffset = TUNER_REG_BYTES-TUNER_DIGTUNE_LEN;

    g_regs.lnaagc.data.all = 0;
    g_regs.lnaagc.size = TUNER_LNAAGC_LEN;
	g_regs.lnaagc.dataoffset = TUNER_REG_BYTES-TUNER_LNAAGC_LEN;

    g_regs.data_addr.data.all = 0;
    g_regs.data_addr.size = TUNER_DATA_ADDR_LEN;
	g_regs.data_addr.dataoffset = TUNER_REG_BYTES-TUNER_DATA_ADDR_LEN;

    g_regs.regtest.data.all = 0;
    g_regs.regtest.size = TUNER_REGTEST_LEN;
	g_regs.regtest.dataoffset = TUNER_REG_BYTES-TUNER_REGTEST_LEN;

    g_regs.vcotest.data.all = 0;
    g_regs.vcotest.size = TUNER_VCOTEST_LEN;
	g_regs.vcotest.dataoffset = TUNER_REG_BYTES-TUNER_VCOTEST_LEN;

    g_regs.lnagain.data.all = 0;
    g_regs.lnagain.size = TUNER_LNAGAIN_LEN;
	g_regs.lnagain.dataoffset = TUNER_REG_BYTES-TUNER_LNAGAIN_LEN;

    g_regs.refagc.data.all = 0;
    g_regs.refagc.size = TUNER_REFAGC_LEN;
	g_regs.refagc.dataoffset = TUNER_REG_BYTES-TUNER_REFAGC_LEN;
 

    /* Next, set the address values for the control registers. */
    g_regs.powerdown.address = TUNER_POWERDOWN;
    g_regs.refosc.address = TUNER_REFOSC;
    g_regs.refdiv.address = TUNER_REFDIV;
    g_regs.mixref.address = TUNER_MIXREF;
    g_regs.resetso.address = TUNER_RESETSO;
    g_regs.lo1.address = TUNER_LO1;
    g_regs.lo2.address = TUNER_LO2;
    g_regs.circuitadj.address = TUNER_CIRCUITADJ;
    g_regs.test.address = TUNER_TEST;
    g_regs.digtune.address = TUNER_DIGTUNE; /* The main address is the same */
    g_regs.lnaagc.address = TUNER_LNAAGC;
    g_regs.data_addr.address = TUNER_DATA_ADDR;
    g_regs.regtest.address = TUNER_REGTEST;
    g_regs.vcotest.address = TUNER_VCOTEST;
    g_regs.lnagain.address = TUNER_LNAGAIN;
    g_regs.refagc.address = TUNER_REFAGC;
}


/****************************************************************************
* Function Name: select_RFagc_IFout
*
* Description:
* This function selects the RF AGC input and the IF output for the MC44S803.
* This function does not work with the MC44S802A.
*
* Parameters: none.
* Parameters:
*    rf_agc_sel: indicates which of the tuner's two RF AGC inputs to use.
*    if_out_sel: indicates which of the tuner's two IF output ports to use.
*
* Input conditions:
*    A MC44S803 is being used.
*
* Output conditions:
*    The RF AGC input and IF output are selected per the parameters.
*
* Return values:
*      >0 : an I2C error occured.  The returned value is an I2C error code.
*      0 : the data was sucesfully transmited.
*
*****************************************************************************/
int MC44S803_select_RFagc_IFout ( int rf_agc_sel, int if_out_sel)
{
	int ret;

	g_regs.circuitadj.data.bits.OutSel = if_out_sel;
	ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.circuitadj));
	if (ret == I2C_OK) {
	    g_regs.lnaagc.data.bits.AgcSel = rf_agc_sel;
	 	ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.lnaagc));
	 }
	return (ret);
}


/****************************************************************************
* Function Name: tuner_init
*
* Description:
* This function intializes the internal data structures and the tuner
* hardware.  The prescaler for the digital tune state machine is set based
* on the reference clock frequency, which is defined in tuner_settings.h.
* The RF AGC input and IF output are selected based on the parameters
*
* Parameters:
*    tuner_ver: indicates if a MC44S802A or MC44S803 is being used.
*    rf_agc_sel: indicates which of the tuner's two RF AGC inputs to use.
*    if_out_sel: indicates which of the tuner's two IF output ports to use.
*
* Return type: integer
* Return values:
*      >0 : an I2C error occured.  The returned value is an I2C error code.
*      0 : the data was sucesfully transmited.
*
* Input conditions:
* I2C port initalized.
*
* Output conditions:
* The tuner is reset.  The register MixRef, Circuit Adjust, LNAGain, and
* digital tune are initalized.  The RF AGC input and IF output are selected.
*
*****************************************************************************/
int MC44S803_tuner_init (int rf_agc_sel, int if_out_sel)
{
    int ret;
    
	MC44S803_tuner_init_globals ();
 
    /* Reset the tuner */
 	g_regs.resetso.data.bits.RS = RESETSO_RS_RESET;
 	ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.resetso));
 	if (ret)
 		return (ret);
 
    /* Take the part out of reset */
 	g_regs.resetso.data.bits.RS = RESETSO_RS_OPERATE;
 	ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.resetso));
 	if (ret)
 		return (ret);
 
	/* Turn on the chip */
	ret = MC44S803_set_powerdown (POWERDOWN_ON);
 	if (ret)
 		return (ret);
 
	/* Wait about 20 mSec for the reference oscillator to stablize */
	msleep(20);//SysDelay(20); /* wait (20) */]
	
	
	/* Initialize the Reference Oscillator Register */
 	if (g_ref_freq < 16000000) {
 		g_regs.refosc.data.bits.OscSel = REFOSC_OSCSEL_4M;
 	} else {
 		g_regs.refosc.data.bits.OscSel = REFOSC_OSCSEL_25M;
 	}
 	g_regs.refosc.data.bits.test = REFOSC_TEST;
 	ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.refosc));
  	if (ret)
 		return (ret);

	/* Initialize the Mixer Refernce Register to the default values. */
	g_regs.mixref.data.bits.Mix1 = 0x3;
	g_regs.mixref.data.bits.QuadGenEn = 1;
	g_regs.mixref.data.bits.ImageReject = 0xF;
	g_regs.mixref.data.bits.BufIO = 0;
	g_regs.mixref.data.bits.BufGain = 0;
	g_regs.mixref.data.bits.TriState = 1;
	g_regs.mixref.data.bits.OscScr = 0;
	g_regs.mixref.data.bits.Mux4 = 0;
	g_regs.mixref.data.bits.Mux3 = 0;
	g_regs.mixref.data.bits.R3Div = MIXREF_R3DIV_1;
 	ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.mixref));
 	if (ret)
 		return (ret);
 
	/* initialize the Circuit Adjust Register */
	/* Set control bits and clear IF & RF Detectors */
	g_regs.circuitadj.data.bits.IFDectClear = 1;
	g_regs.circuitadj.data.bits.RFDectClear = 1;
	g_regs.circuitadj.data.bits.IFAgcAmpLP = CIRCUITADJ_IFAGCLP_NOMINAL;
	g_regs.circuitadj.data.bits.rsvd1 = 0;
	g_regs.circuitadj.data.bits.PostAmp2LP = CIRCUITADJ_POSTAMP2LP_NOMINAL;
	g_regs.circuitadj.data.bits.Mix1LP = CIRCUITADJ_MIXLP_NOMINAL;		/* CIRCUITADJ_MIXLP_HIGH */
	g_regs.circuitadj.data.bits.PreAmp1LP = CIRCUITADJ_PREAMP1LP_NOMINAL;
	g_regs.circuitadj.data.bits.OutSel = if_out_sel;
	g_regs.circuitadj.data.bits.IFAmpGain = CIRUITADJ_IFAMPGAIN_12DB;	/* CIRUITADJ_IFAMPGAIN_6DB */
	g_regs.circuitadj.data.bits.rsvd2 = 0x3;
	g_regs.circuitadj.data.bits.PostAmp2Gain = CIRUITADJ_POSTAMP2GAIN_83DB;
	g_regs.circuitadj.data.bits.rsvd3 = 0x0;
	g_regs.circuitadj.data.bits.PreAmp1Gain = CIRCUITADJ_PREAMP1_GAIN_1DB;
	g_regs.circuitadj.data.bits.rsvd4 = 1;
	ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.circuitadj));
	if (ret)
		return (ret);
 			
	/* Put the peak detectors in operational mode */
	g_regs.circuitadj.data.bits.IFDectClear = 0;
	g_regs.circuitadj.data.bits.RFDectClear = 0;

	ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.circuitadj));
	if (ret)
		return (ret);

 	/* Now, Initialize the Digital Tune prescale register */
 	g_regs.digtune.data.bits.xo.dtaddr = DIGTUNE_DTADDR_XO;
 	g_regs.digtune.data.bits.xo.rw = DIGTUNE_WRITE;
 	
 	
 	if (g_ref_freq  < 4200000) {
 		g_regs.digtune.data.bits.xo.xod = DIGTUNE_XOD_4_2_MHZ;
 	} else if (g_ref_freq < 8200000) {
 		g_regs.digtune.data.bits.xo.xod = DIGTUNE_XOD_8_2_MHZ;
 	} else if (g_ref_freq < 16500000) {
 		g_regs.digtune.data.bits.xo.xod = DIGTUNE_XOD_16_5_MHZ;
 	} else {
 		g_regs.digtune.data.bits.xo.xod = DIGTUNE_XOD_28_0_MHZ;
 	}

 	ret =  MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.digtune));
 	if (ret)
 		return (ret);
  	
 	/* Setup the LNA to use digital AGC. Select the RF AGC input. */
    g_regs.lnaagc.data.bits.A0 = 0;
    g_regs.lnaagc.data.bits.AgcSel = rf_agc_sel;
    g_regs.lnaagc.data.bits.Lna0 = 1;
    g_regs.lnaagc.data.bits.Bias = 0;
 
    g_regs.lnaagc.data.bits.ReadEn = 1;
    g_regs.lnaagc.data.bits.HighAtten = LNAAGC_HIGHATTEN_NORMAL;
    g_regs.lnaagc.data.bits.AnDig = LNAAGC_ANDIG_ANALOG;
    g_regs.lnaagc.data.bits.HlGrEn = LNAAGC_HIGHLIN_DISABLE;
    g_regs.lnaagc.data.bits.rsvd2 = 0;
    g_regs.lnaagc.data.bits.A2 = 1;
    g_regs.lnaagc.data.bits.A1 = 1;
    g_regs.lnaagc.data.bits.NormalAgc = 0;
 	ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.lnaagc));
  	if (ret)
 		return (ret);
 	
 	return (OK);
}


/****************************************************************************
* Function Name: tune_mc44s803
*
* Description:
*   This routine tunes the MC44S803 to the desired channel by setting the
* first and second LOs, and then digtune() to initate the automatic PLL
* fine tuning state machine.
*
* Parameters:
*	freq: a double percision floating point number representing the center
*      frequency of the desired channel, in Hz.
*
* Input Conditions:
*    The I2C bus has been initialized.
*    tuner_init has been succesfully called.
*
* Return type: integer
* Return values:
*      >0 : an I2C error occured.  The returned value is an I2C error code.
*      0 : the data was sucesfully transmited.
*
*
*
*****************************************************************************/
int MC44S803_tune (int freq)
{
    int ret;
    int channel_index;
    
    ret = OK;

    channel_index = MC44S803_locate_settings (freq);

    if (channel_index != FREQ_TO_LOW) {
    
//    	printf ("Freq:%d, ", freq);
    	
        /* Update the control registers: refdiv, lo1, lo2 */
        g_regs.refdiv.data.all = g_tuner_settings_table[channel_index].refdiv;
 		ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.refdiv));
 	 	if (ret)
 			return (ret);
// 		printf ("r1:%d, r2:%d, ", g_regs.refdiv.data.bits.R1Div, g_regs.refdiv.data.bits.R2Div);

        g_regs.lo1.data.all = g_tuner_settings_table[channel_index].lo1;
 		ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.lo1));
 	 	if (ret)
 			return (ret);
// 		printf ("lo1:%d, ", g_regs.lo1.data.bits.n);

        g_regs.lo2.data.all = g_tuner_settings_table[channel_index].lo2;
		ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.lo2));
 	 	if (ret)
 			return (ret);
//		printf ("lo2:%d, ", g_regs.lo2.data.bits.n);
		
        /* Last step is to update the digi-tune register */
        g_regs.digtune.data.all = g_tuner_settings_table[channel_index].digtune_lo1;
		ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.digtune));
 	 	if (ret)
 			return (ret);
// 		printf ("lo1ref:%d, ", g_regs.digtune.data.bits.lo1.lo1ref);
 
        g_regs.digtune.data.all = g_tuner_settings_table[channel_index].digtune_lo2;
		ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.digtune));
 	 	if (ret)
 			return (ret);
//    	printf ("lo2ref:%d", g_regs.digtune.data.bits.lo2.lo2ref);

    } else {
        ret = FREQ_TO_LOW;
    }

    return (ret);

}

/****************************************************************************
* Function Name:mc44s803_clear_dectectors
*
* Description:
*   This routines clears the RF and IF power detectors.
*
* Parameters:
*   None.
*
* Return type: integer
* Return values:
*      >0: an I2C error occured.  The returned value is an I2C error code.
*      0: proper I2C communication occured.
*
*****************************************************************************/
int mc44s803_clear_dectectors ()
{
    int ret;
    
    /* Set the bit to clear the RF & IF Dectectors */
    g_regs.circuitadj.data.bits.IFDectClear = 1;
    g_regs.circuitadj.data.bits.RFDectClear = 1;

    ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.circuitadj));
    if (ret)
        return (ret);
    
    /* Put the detectors back in operational mode */
    g_regs.circuitadj.data.bits.IFDectClear = 0;
    g_regs.circuitadj.data.bits.RFDectClear = 0;

    ret = MC44S803_i2c_xmit_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.circuitadj));
    return (ret);
}

/****************************************************************************
* Function Name: mc44s803_read_avg_rf_dectector
*
* Description:
*   This routines reads the RF detector multiple times and averages
* the result.  The number of reads is set in tuner_settings.h
*
* Parameters:
*	result: a pointer to the location to place the result.
*
* Return type: integer
* Return values:
*      >0: an I2C error occured.  The returned value is an I2C error code.
*      0: proper I2C communication occured.
*
*****************************************************************************/
int mc44s803_read_avg_rf_dectector (int *result)
{
	int	i;
	int ret;
	int gray_rf_pwr;
	int rf_pwr;

	i = 0;
	rf_pwr = 0;

	while (i< NUM_DETECTOR_AVG) {
		/* first, clear the RF & IF Dectectors */
		ret = mc44s803_clear_dectectors ();
		if (ret)
			return (ret);
		
		/* Now, read the detectors in the LNA gain data register */
		ret =  MC44S803_i2c_read_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.lnagain));
		if (ret)
			return (ret);
		gray_rf_pwr = g_regs.lnagain.data.bits.rfpwr;

		/* convert power from gray code to standard and sum */
		rf_pwr += g_gray_table[gray_rf_pwr];

		i++;
	}
	
	/* Round prior to integer divide */
	rf_pwr += (NUM_DETECTOR_AVG/2);
	
	/* Aveage */
	rf_pwr = rf_pwr / NUM_DETECTOR_AVG;
	
	/* Save the result */
	*result = rf_pwr;

	return (ret);
}



/****************************************************************************
* Function Name: mc44s803_read_avg_if_dectector
*
* Description:
*   This routines reads the IF detector multiple times and averages
* the result.  The number of reads is set in tuner_settings.h
*
* Parameters:
*	result: a pointer to the location to place the result.
*
* Return type: integer
* Return values:
*      >0: an I2C error occured.  The returned value is an I2C error code.
*      0: proper I2C communication occured.
*
*****************************************************************************/
int mc44s803_read_avg_if_dectector (int *result)
{
	int	i;
	int ret;
	int gray_if_pwr;
	int if_pwr;

	i = 0;
	if_pwr = 0;
	
	while (i< NUM_DETECTOR_AVG) {
		/* first, clear the RF & IF Dectectors */
		ret = mc44s803_clear_dectectors ();
		if (ret)
			return (ret);
		
		/* Now, read the detectors in the LNA gain data register */
		ret =  MC44S803_i2c_read_tuner_reg ((GENERIC_STRUCT_T *) &(g_regs.lnagain));
		if (ret)
			return (ret);
		gray_if_pwr = g_regs.lnagain.data.bits.ifpwr;
	
		/* convert power from gray code to standard and sum */
		if_pwr += g_gray_table[gray_if_pwr];
		
		i++;
	}
	
	/* Round prior to integer divide */
	if_pwr += (NUM_DETECTOR_AVG/2);
	
	/* Aveage */
	if_pwr = if_pwr / NUM_DETECTOR_AVG;
	
	/* Save the result */
	*result = if_pwr;

	return (ret);
}

/****************************************************************************
* Function Name: mc44s803_rf_adjust_lna_gain
*
* Description:
*   This routines adjusts the front end LNA gain to to keep the power
* measured by the RF detector within the prefered operating range.
* It reads the RF power detector and adjusts the LNA gain until the RF
* detector is in the prefered range, or railed.
*
* Parameters:
*   None.
*
* Return type: integer
* Return values:
*      >0: an I2C error occured.  The returned value is an I2C error code.
*      0: proper I2C communication occured.
*
*****************************************************************************/
/*int mc44s803_rf_adjust_lna_gain ()
}
*/



/****************************************************************************
* Function Name: mc44s803_if_adjust_lna_gain
*
* Description:
*   This routines adjusts the front end LNA gain to to keep the power
* measured by the IF detector within the prefered operating range.
* It reads the IF power detector and adjusts the LNA gain until the RF
* detector is in the prefered range, or railed.
*
* This routine should only be called after the RF agc loop has been run.
*
* Parameters:
*   None.
*
* Return type: integer
* Return values:
*      >0: an I2C error occured.  The returned value is an I2C error code.
*      0: proper I2C communication occured.
*
*****************************************************************************/
/*int mc44s803_if_adjust_lna_gain ()
{
}
*/

/****************************************************************************
* Function Name: mc44s803_adjust_lna_gain
*
* Description:
*   This routines adjusts the front end LNA gain to to keep the power
* measured by the RF & IF power detectors within the prefered operating
* range.  It first reads the RF power detector and adjusts the LNA gain
* until the RF detector is in the prefered range, or railed.  Then, it
* repeats this process using the IF detector.
*
* Parameters:
*   None.
*
* Return type: integer
* Return values:
*      >0: an I2C error occured.  The returned value is an I2C error code.
*      0: proper I2C communication occured.
*
*****************************************************************************/
/*int mc44s803_adjust_lna_gain ()
{
    int ret;    
        
    ret = mc44s803_rf_adjust_lna_gain ();
    if (ret)
        return (ret);
    
    ret = mc44s803_if_adjust_lna_gain ();
    
    return (ret);   
}
*/



